﻿/*
VERSION:		2.6
2.6:				changed _visibility to prevent the charset from being visible when loading from an external file
2.5:				a temporary image is used to reduce flicker
2.4:				setPose acts differently than past versions  (uses animType instead of animYoyo)
	
DESCRIPTION:
	Pros:		A single image can be used,  and can contain many poses.
				You can specify the:  size,  number of poses,  number animation frames,  number of directions.
				Animation speed can be controlled.
				Animation can occelate back & forth,  or loop.
	Cons:		All poses must have the same number of animation frames.
	
NOTE:
	This sprite system places the character's feet at the origin point.	(Feet:  1/2 image width, full image height)
	
	
USAGE:
	mySprite = makeSprite( this, "marle1_mc" );			// minimum usage
	makeSprite( this, "marle1_mc", "mySprite_mc", 10, 24, 32, 3, 4, 4, 2, 3 );			// maximum usage		(rm2k settings)
	
	makeSprite( target, newCharset, [newName], [depth], [spriteWidth], [spriteHeight], [maxAnimFrames], [maxDirections], [maxPoseColumns], [maxPoseRows], [newAnimSpeed] )
	
PUBLIC FUNCTIONS:
	setCharset( newCharset, width, height, frames, animType, animDelay )		Changes the charset image
	setPose( poseNumber, newAnimType )																			Changes the pose, and the animation mode.  (true = back & forth,  false = loop)
	setDirection( newDirection )																						Changes the direction.
	setAnim( true )																													Turns animation on / off  (boolean)
	setFrame( newFrame )																										Shows the specified frame of animation.
	setSpeed( newDelay )																										Changes the animation speed.
	setAll( charset, pose, direction, anim, frame, animSpeed )							Set charset, pose, direction, anim, frame, speed
	
	
PUBLIC SETTINGS:
	
	direction		The direction that the sprite is facing.		(number)  0=up  1=right  2=down  3=left
	pose				The pose that the sprite is in.
	anim				Whether or not the sprite is animating.
	animType		How to animate:		yoyo, loop, once
	animFrame		Which frame of animation is currently shown.
	animSpeed		How fast the sprite animates.		(How many frames to wait between animation frames)
	charset			The charset image that this sprite is using.
	
	imageOffsetX	The horizontal offset of the image.		(default is half the sprite's width to center the sprite horizontally)
	imageOffsetY	The vertical offset of the image.			(default is the very bottom and then up by 1/8 of the sprite's height, to center their feet on their given location)
	
	
DEPENDANCIES:
	nextDepth.as
*/



function makeSprite( target, newCharset, newName, depth, spriteWidth, spriteHeight, maxAnimFrames, maxDirections, maxPoseColumns, maxPoseRows, newAnimSpeed )
{
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// resolve optional parameters
	var newName = (newName) ? newName : String("sprite_"+Math.round(Math.random() * 99999) );		// default:	sprite_94627
	#include "nextDepth.as"
	var depth = (depth) ? depth : nextDepth(target);												// default depth = 1
	var spriteWidth = (spriteWidth) ? spriteWidth : 24;					// mask is 24 pixels wide
	var spriteHeight = (spriteHeight) ? spriteHeight : 32;					// mask is 32 pixels tall
	var maxAnimFrames = (maxAnimFrames) ? maxAnimFrames : 3;
	var maxDirections = (maxDirections) ? maxDirections : 4;			// up, right, down, left
	var maxPoseColumns = (maxPoseColumns) ? maxPoseColumns : 4;
	var maxPoseRows = (maxPoseRows) ? maxPoseRows : 2;
	var newAnimSpeed = (newAnimSpeed) ? newAnimSpeed : 3;	// wait 3 frames between updates
	
	// CREATE
	// // This Sprite movieClip
	var _this = target.createEmptyMovieClip( newName, depth );
	
	// // image_mc
	//_this.attachMovie( newCharset, "image_mc", 1);
	_this.createEmptyMovieClip( "image_mc", 1 );
	_this.image_mc._x = _this.image_mc._y = 9999;
	
	// // mask_mc
	_this.createEmptyMovieClip( "mask_mc", 2);
	_this.mask_mc.beginFill( 0x000000, 0);
	_this.mask_mc.moveTo(0, 0);		// start
	_this.mask_mc.lineTo(16, 0);		// right
	_this.mask_mc.lineTo(16, 16);	// down
	_this.mask_mc.lineTo(0, 16);		// left
	_this.mask_mc.lineTo(0, 0);		// up
	_this.mask_mc.endFill();					// stop
	_this.image_mc.setMask( _this.mask_mc );
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// PUBLIC FUNCTIONS
	_this.setDirection = function( newDirection )
	{
		if ( isNaN(newDirection) )
		{
			switch (newDirection)
			{
				case "up":
					newDirection = 0;
				break;
				case "right":
					newDirection = 1;
				break;
				case "down":
					newDirection = 2;
				break;
				case "left":
					newDirection = 3;
				break;
			}// switch
		}// if isNaN
		
		_this.direction = newDirection;
	}// setDirection()
	
	
	
	_this.setPose = function( newPose, newAnimType )
	{
		_this.pose = newPose;
		
		_this.animType = (newAnimType!=undefined) ? newAnimType : _this.animType;		// use new setting if specified.
	}// setPose
	
	
	
	_this.setAnim = function( isAnimating )
	{
		// when stopping, default to frame 1  (Convenient when stopping a walk)
		if(_this.anim==true  &&  isAnimating==false)
		{// if:  animation is being stopped
			_this.animFrame = _this.getFirstFrame();
		}// if:  animation is being stopped
		
		_this.anim = isAnimating;
	}// setAnim
	
	
	
	_this.setSpeed = function( newSpeed )
	{
		newSpeed = Math.round(newSpeed);		// removes decimals
		if (newSpeed <= 0  ||  newSpeed == undefined){ newSpeed = 1 };		//	enforce a valid range.	(prevents divide by zero)
		_this.animSpeed = newSpeed;
	}// setSpeed()
	
	
	
	_this.setFrame = function( newFrame )
	{
		_this.animFrame = newFrame;
	}// setFrame()
	
	
	
	_this.setCharset = function( newCharset, newWidth, newHeight, newFrames, newAnimType, newSpeed )
	{
		_this.charset = newCharset;
		
		if(newWidth)
			_this.spriteWidth = newWidth;
		if(newHeight)
			_this.spriteHeight = newHeight;
		if(newFrames)
			_this.maxAnimFrames = newFrames;
		if(newAnimType)
			_this.animType = newAnimType;
		if(newSpeed!=undefined)
			_this.setSpeed(newSpeed);
		
		_this.setupCharset();		// immediately update, to avoid flicker
		if(newWidth || newHeight)
			_this.setupMask();			// update the mask if the size changed
		_this.updateImage();			// don't wait for the animation delay to show the new pose
	}// setCharset()
	
	
	
	_this.setAll = function( charset, pose, direction, anim, frame, speed )
	{
		if(charset!=undefined)
			_this.setCharset( charset );
		
		if(pose!=undefined)
			_this.setPose( pose );
		
		if(direction!=undefined)
			_this.setDirection( direction );
		
		if(anim!=undefined)
			_this.setAnim( anim );
		
		if(frame!=undefined)
			_this.setFrame( frame );
		
		if(speed!=undefined)
			_this.setSpeed( speed );
	}// setAll()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// SUPPORT FUNCTIONS
	
	_this.getFirstFrame = function()
	{
		if(_this.maxAnimFrames > 1  &&  _this.animType=="yoyo")
		{// if:  rpgMaker style
			return 1;
		}// if:  rpgMaker style
		else
		{// there's only 1 frame
			return 0;
		}// there's only 1 frame
	}// getFirstFrame()
	
	
	
	_this.updateDirection = function()
	{
		_this.direction %= _this.maxDirections;		// Limit direction changes to be within bounds.
		_this.y -= _this.direction * _this.spriteHeight;
	}// updateDirection()
	
	
	
	_this.updatePose = function()
	{
		var maxPoses = _this.maxPoseColumns * _this.maxPoseRows;
		_this.pose %= maxPoses;		// Limit poses to be within bounds.
		
		var thisColumn = _this.pose % _this.maxPoseColumns;
		var columnWidth = _this.spriteWidth * _this.maxAnimFrames;
		var thisRow = Math.floor(_this.pose / _this.maxPoseColumns);
		var rowHeight = _this.spriteHeight * _this.maxDirections;
		_this.x -= (thisColumn * columnWidth);
		_this.y -= (thisRow * rowHeight);
	}// updateDirection()
	
	
	
	_this.updateAnimation = function()
	{
		if (_this.animType=="yoyo")
		{	// animate back & forth
			if (_this.animFrame >= _this.maxAnimFrames-1)
			{	// if this is the last frame
				_this.animDirection = -1;		// animate backwards
			}// if	>= maxAnimFrames
			else if (_this.animFrame <= 0)
			{	// if this is the first frame
				_this.animDirection = 1;		// animate forwards
			}// if	<= 0
		}// if yoyo
		else if (_this.animType=="loop")
		{// if:  animate forward only
			_this.animFrame %= (_this.maxAnimFrames);
			_this.animDirection = 1;		// animate forwards
		}// if:  animate forward only
		else if (_this.animType=="once")
		{// if:  animate once
			if(_this.animFrame < _this.maxAnimFrames-1){
				_this.animDirection = 1;		// animate forwards
			}else{
				_this.animDirection = 0;		// stop on the last frame
			}
		}// if:  animate once
		
		if(_this.firstFrameShown)
		{// if:  1st frame has already been shown
			_this.x -= _this.animFrame * _this.spriteWidth;
		
			if(_this.anim  &&  _this.maxAnimFrames>1)
			{// if:  sprite is animating  &  anim frames haven't exceeded their maximum
				_this.animFrame += _this.animDirection;
			}// if:  sprite is animating  &  anim frames haven't exceeded their maximum
		}// if:  1st frame has already been shown
		
		_this.firstFrameShown = true;		// this function does nothing on the 1st frame
	}// updateDirection()
	
	
	
	_this.setupMask = function()
	{
		_this.image_mc.setMask( _this.mask_mc );
		_this.mask_mc._width = _this.spriteWidth;
		_this.mask_mc._height = _this.spriteHeight;
		_this.mask_mc._x = _this.imageOffsetX;
		_this.mask_mc._y = _this.imageOffsetY;
	}// setupMask()
	
	
	
	_this.setupCharset = function()
	{
		if(_this.charset != _this.oldCharset)
		{// if:  charset changes
			// take a snapshot to retain an image and prevent flickering
			delete _this.last_pic;		// remove previous snapshot image
			_this.last_pic = new flash.display.BitmapData( _this.mask_mc._width, _this.mask_mc._height, true, 0x00000000 );
			var copyPosition = new flash.geom.Matrix();
			var xCopy = -_this.imageOffsetX;
			var yCopy = -_this.imageOffsetY;
			copyPosition.translate(xCopy, yCopy);		// reposition the copy area relative to the origin
			_this.last_pic.draw( _this, copyPosition );
			removeMovieClip(_this.last_mc);		// remove previous snapshot
			_this.createEmptyMovieClip( "last_mc", 0 );
			_this.last_mc.attachBitmap( _this.last_pic, 0 );
			_this.last_mc._x = -xCopy;
			_this.last_mc._y = -yCopy;
			var removeLastPic = function()
			{
				removeMovieClip(_this.last_mc);		// remove placeholder image
			}// removeLastPic()
			//setTimeout( updateAfterEvent, 0 );		// forces updateAfterEvent() to run
			
			
			// image_mc
			_this.image_mc._x = _this.image_mc._y = 9999;
			var image_pic = flash.display.BitmapData.loadBitmap( _this.charset );
			if(image_pic.width > -1)
			{// linkage succeeded
				// attach image
				_this.image_mc.attachBitmap( image_pic, 0, true, false );
				// update display position
				_this.updateImage();
				_this.animFrame = _this.getFirstFrame();
				//setTimeout( removeLastPic, 1 );		// updateAfterEvent()  prevents this from working
				removeLastPic();
				//setTimeout( updateAfterEvent, 0 );
			}// if:  linkage succeeded
			else
			{// no linkage available
				// load external file
				_this.loader = new MovieClipLoader();
				_this.image_mc._alpha = 0;
				_this.loader.loadClip( _this.charset, _this.image_mc );
				
				_this.loader.onLoadInit = function()
				{// if:  image loaded
					// update display position
					_this.updateImage();
					_this.animFrame = _this.getFirstFrame();
					//setTimeout( removeLastPic, 33 );		// updateAfterEvent()  prevents this from working
					removeLastPic();
					_this.image_mc._alpha = 100;
					//updateAfterEvent();
					//setTimeout( updateAfterEvent, 0 );
				}// if:  image loaded
				
				_this.loader.onLoadError = function()
				{// no file available
					// attach movieClip
					_this.image_mc.removeMovieClip();
					_this.attachMovie( _this.charset, "image_mc", 0 );
					_this.image_mc.setMask( _this.mask_mc );
					removeLastPic();
					//setTimeout( updateAfterEvent, 0 );
					
					// clean-up
					delete _this.loader;
				}// if:  no file available
			}// if:  no linkage available
			
			
			
			// remember in case the setting changes
			_this.oldCharset = _this.charset;
		}// if:  charset changes
	}// setupCharset()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// EXTRA FEATURES
	
	// This makes the specified sprite look at another sprite.
	_this.lookAt = function( sprite2 )
	{
		var sprite1 = _this;
		// get a unit vector
		var xDiff = sprite2._x - sprite1._x;
		var yDiff = sprite2._y - sprite1._y;
		var dist = Math.sqrt( xDiff*xDiff + yDiff*yDiff );
		var xUnit = xDiff / dist;
		var yUnit = yDiff / dist;
		
		// compare x & y
		if( Math.abs(yUnit) > Math.abs(xUnit) )
		{
			// vert
			if( yUnit < 0)
			{
				// up
				_this.setDirection( "up" );
			}else{
				// down
				_this.setDirection( "down" );
			}// if:  up or down
		}else{
			// horz
			if( xUnit < 0 )
			{
				// left
				_this.setDirection( "left" );
			}else{
				// right
				_this.setDirection( "right" );
			}// if:  left or right
		}// if:  vert or horz
	}// lookAt()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	//  SETTINGS
	
	// Default charset settings
	_this.spriteWidth = (spriteWidth) ? spriteWidth : 16;		// default:		16 px wide
	_this.spriteHeight = (spriteHeight) ? spriteHeight : 16;		// default:		16 px tall
	_this.maxAnimFrames = (maxAnimFrames) ? maxAnimFrames : 1;		// default:		1 frame of animation
	_this.maxDirections = (maxDirections) ? maxDirections : 1;		// default:		1 direction
	_this.maxPoseColumns = (maxPoseColumns) ? maxPoseColumns : 1;		// default:		1 pose horizontally
	_this.maxPoseRows = (maxPoseRows) ? maxPoseRows : 1;		// default:		1 pose vertically
	_this.setCharset(newCharset);
	_this.animDirection = 1;
	_this.setSpeed(newAnimSpeed);
	
	// // Initialize Private Variables
	_this.x 								= 0;						// This offsets the sprite image horizontally.
	_this.y 								= 0;						// This offsets the sprite image vertically.
	_this.imageOffsetX			= -1*(spriteWidth/2);	// -12		// This offsets the image to place the registration point in the middle.
	_this.imageOffsetY 			= -1*(spriteHeight - Math.round(spriteHeight/8));			// -28		// This offsets the image to place the registration point 8 pixels above the feet.
	_this.animDelay					= 0;						// This counter delays the animation until it hits animSpeed, then loops.
	
	// // Default Public Settings
	_this.direction		= 	0;
	_this.pose				= 	0;
	_this.anim 				= true;
	_this.animType		= "yoyo";
	_this.animFrame 	= _this.getFirstFrame();
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// LOOP
	_this.updateImage = function()
	{
		// Reset
		_this.x = 0;
		_this.y = 0;
		
		// Set direction
		_this.updateDirection();
		
		// Set pose
		_this.updatePose();
		
		// Do animation
		_this.updateAnimation();
		
		
		// Set the image
		_this.image_mc._x = _this.x + _this.imageOffsetX;
		_this.image_mc._y = _this.y + _this.imageOffsetY;
	}// updateImage()
	
	
	
	_this.onEnterFrame = function()
	{
		// Set the charset
		_this.setupCharset();
		
		// Setup the Mask		(this loops in case settings change)
		_this.setupMask();
		
		// Do delay
		if (_this.animDelay == 0)
		{
			_this.updateImage();
		}// if:		animDelay = 0
		_this.animDelay++;
		_this.animDelay %= _this.animSpeed;
	}// onEnterFrame()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// OUTPUT
	return _this;
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
}// makeSprite()